/**
 * This grammar was written for antlr.
 * The Lexer and Parser are generated from this grammar with antlr.
 * 
 * antlr uses the BSD license:
 * 
 * [The BSD License]
 * Copyright (c) 2003-2008, Terence Parr
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
 *   - Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 *   - Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation 
 * 		and/or other materials provided with the distribution.
 *   - Neither the name of the author nor the names of its contributors may be used to endorse or promote products derived from 
 *		this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */ 

grammar Data; 

@header { 
package de.lmu.ifi.pst.mdd4soa.transform.impl.uml2iom.parser;
import de.lmu.ifi.pst.mdd4soa.transform.exception.TransformationException;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Assignment;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.CompositeElement;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.DataHandling;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Declaration;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Expression;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Field;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Handler;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.ISMFactory;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.LeftHandSideExpression;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Literal;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Operation;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.OperationType;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.RightHandSideExpression;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.ServiceActivity;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.type.Type;
import de.lmu.ifi.pst.mdd4soa.transform.impl.iom.Variable;
import de.lmu.ifi.pst.mdd4soa.transform.impl.uml2iom.util.TypeResolver;
/**
 * The parser generated by antlr from the grammar Data.g  
 * @author andreas rogge-solti
 */
} 

@lexer::header {package de.lmu.ifi.pst.mdd4soa.transform.impl.uml2iom.parser;
/**
 * The lexer generated by antlr from the grammar Data.g
 * @author andreas rogge-solti
 */
}

@members {

	// ************************************************************
    // ******************** Custom methods ************************
    // ************************************************************

   	/**
   	 * The current scope
   	 */
   	private ServiceActivity currentScope;

   	/**
   	 * Tries to retrieve a scope, that is already defined in the context.<br>
   	 * <br>
   	 * Usage in a data-handling node:<br>
   	 * ::scopeName.varName<br>
   	 * At the time you reference a <i>scopeName</i> in a data-handling node, you
  	 * must have had a scope before, which has the name <i>scopeName</i>.
   	 * 
   	 * @param name
   	 *            the name of the scope to be found
   	 * @return {@link ServiceActivity} with the name matching the "name"-parameter.
   	 * @throws TransformationException
   	 *             , if no scope is found with that name. This means that: -
   	 *             There is no such scope at all - or there is one, but it was
   	 *             not yet processed at the time of processing the data handling
   	 *             node.
  	 */

   	private ServiceActivity getScope(String name) {
   		try {
   			return currentScope.getScope(name);
   		} catch (TransformationException tre) {
   			throw new RuntimeException("Antlr cannot throw custom exceptions...", tre);
  		}
   	}

   	/**
   	 * This returns the current scope and is used every time a plain variable is
   	 * referenced (or declared).<br>
   	 * Usage:<br>
   	 * varName.someThing<br>
   	 * This references a variable named <i>varName</i> in the current Scope.
   	 * (which is omitted)<br>
  	 * This resembles the usage of the "this" keyword in Java, that refers to
   	 * the "scope" of the current Object.
   	 * 
   	 * @return {@link ServiceActivity} the current scope of the data handling node.
   	 * @throws TransformationException(RuntimeException)
   	 *             This can occur, when a data handling node is placed somewhere
   	 *             outside of scopes (i.e. there is no surrounding scope in the
   	 *             UML-model).
   	 */
   	private ServiceActivity getCurrentScope() {
		return currentScope;
   	}

    /**
	 * Returns the variable for the specified scope (a new one, if
	 * not existing yet). The scope should be the direct parent
	 * scope of an activity
	 * 
	 * @param name
	 *            the name of the variable to search the scope for (or to
	 *            create)
	 * @param parent
	 *            the scope to search for variables in.
	 * @return the {@link Variable} of the scope with the name in parameter "<i>name</i>".
	 */
	private Variable getVariable(String name, ServiceActivity parent, boolean scanFirst) {
		Variable v = findVariableInContext(name, parent, scanFirst);

		if (v != null){
			// variable exists already 
			return v;
		} else {
			// not found in scope or handler
    		v = ISMFactory.eINSTANCE.createVariable();
    		v.setName(name);
    		// EMF takes care of the bidirectional relationship. We just set this side.
    		v.setScope(parent);
    		return v;
		}
	}
	
	/**
	 * Searches scopes and handlers for a variable with a specific name.
	 *
	 * @param name the name of the variable to look for
	 * @param current the current composite activity to search in
	 * @param recursive boolean flag indicating, if the search should continue in parent of current scope, 
	 * if no variable was found. 
	 */
	private Variable findVariableInContext(String name, CompositeElement current, boolean recursive){
		if (current instanceof ServiceActivity){
			for (Variable varInScope : ((ServiceActivity)current).getVariables()){
				if (varInScope.getName().equals(name)) {
					return varInScope;
				}
			}
		} else if (current instanceof Handler){
			Variable parameter = ((Handler)current).getParameter();
			if (parameter != null){
				if (parameter.getName().equals(name)){
					return parameter;
				}
			}
		}
		if (recursive){
			CompositeElement parent = current.getParent();
		 	if (parent != null){
				return findVariableInContext(name, parent, recursive);
			}
		}
		return null;
	}

    /**
     * This is called, when a declaration needs to be parsed.
     * There the type of the declared variable is only in string form.
	 * The {@link TypeResolver} class stores all Types specified in 
     * the class diagram and can perform the matching.
     * @param typeName the name of the Type to resolve
     * @return the UML {@link Type} with the same name as the parameter
     */
    private Type resolveType(String typeName) {
    	try {
			return TypeResolver.getInstance().resolveType(typeName);
		} catch (TransformationException e) {
			throw new RuntimeException(e);
		}
    }

   	/**
   	 * Creates an SMM {@link Operation} with two operands and a type.
   	 * @param operand1 the first operand (an {@link Expression})
   	 * @param type the {@link OperationType} for this binary operation
   	 * @param operand2 the second operand (an {@link Expression})
   	 * @return the newly created {@link Operation} 
   	 */
   	private Operation createBinaryOperation(Expression operand1, OperationType type, Expression operand2) {
   		Operation op = createUnaryOperation(operand1, type);
   		op.getOperands().add(operand2);
   		return op;
   	}

   	/**
   	 * Creates an SMM {@link Operation} with one operand and a type.
   	 * @param operand1 the only operand (an {@link Expression})
   	 * @param type the {@link OperationType} for this binary operation
   	 * @return the newly created {@link Operation} 
   	 */
   	private Operation createUnaryOperation(Expression operand1, OperationType type) {
   		Operation op = ISMFactory.eINSTANCE.createOperation();
   		op.setOperationType(type);
   		op.getOperands().add(operand1);
   		return op;
   	}

   	// ************************************************************
   	// **************** End of Custom methods *********************
   	// ************************************************************
} 

/* The root node for multiple assignments */
data[DataHandling dh] 	:	{try{this.currentScope = $dh.getCurrentScope();}catch(TransformationException tre){throw new RuntimeException(tre);}}
			( a=assignment {$dh.getStatements().add($a.value);} | 
			  d=declaration {$dh.getStatements().add($d.value);})*;
/* One assignment with a left-hand and right-had side expression */
assignment returns [Assignment value]	: {$value = ISMFactory.eINSTANCE.createAssignment();}	
			l=leftHandExp[this.currentScope] {$value.setTarget($l.value);} 
			ASSIGN 
			e = rightHandExp[this.currentScope] {$value.setSource($e.value);} SEMICOLON;

declaration returns [Declaration value]	:
			t=typeName 
			e = variable[getCurrentScope(),false]
			{$value = ISMFactory.eINSTANCE.createDeclaration();
			e.setType($t.value);
			$value.setDeclaredVar(e);} SEMICOLON;


scopeName returns [String value] 
	:	SCOPE_REF e = name {$value = $e.value;};
variable[ServiceActivity sc,boolean recursive] returns [Variable value]
	:	e = name {$value = getVariable($e.value,$sc,$recursive);}; 

field [LeftHandSideExpression parent] returns [Field value]
	:	{LeftHandSideExpression currentParent = parent;}
	((
		DOT f=name
		{$value = ISMFactory.eINSTANCE.createField(); 
	 	 $value.setName($f.value);
		 $value.setParent(currentParent); 
		 $value.setType(TypeResolver.getInstance().getTypeOfField($value));
		 currentParent = $value;} 
	)+); 

parExpression returns [RightHandSideExpression value]
	:	'(' rhs=rightHandExp[this.currentScope] ')'
	{$value = $rhs.value;}
	;


leftHandExp[ServiceActivity sc] returns [LeftHandSideExpression value]	
	: {this.currentScope = $sc;}
	(s = scopeName DOT e = variable[getScope(s),false]
	   | e = variable[sc,true])
	{$value = $e.value;}
	(f=field[$value]{$value = $f.value;})?;
	  
rightHandExp[ServiceActivity sc]	returns [RightHandSideExpression value]
	: {this.currentScope = $sc;}
	 (op = conditionalOrOperation {$value = $op.value;});
	 
conditionalOrOperation returns [RightHandSideExpression value]
	:  op2 = conditionalAndOperation {$value = $op2.value; RightHandSideExpression lastOp = $value;} 
	  ((OR | OR2) op2=conditionalAndOperation {$value = createBinaryOperation(lastOp,OperationType.OR, op2); lastOp = $value;})*;

conditionalAndOperation returns [RightHandSideExpression value]
	: op2 = equalityOperation {$value = $op2.value; RightHandSideExpression lastOp = $value;}
	((AND | AND2) op2 = equalityOperation {$value = createBinaryOperation(lastOp,OperationType.AND, op2); lastOp = $value;})*;
	
equalityOperation returns [RightHandSideExpression value]
	: op2 = relationalOperation {$value = $op2.value; RightHandSideExpression lastOp = $value;  OperationType type = null;}
	(((EQUALS2 | EQUALS) op2 = relationalOperation {type = OperationType.EQUALS;}
	| NOT_EQUALS op2 = relationalOperation {type = OperationType.NOT_EQUALS;})
	{$value = createBinaryOperation(lastOp,type, op2); lastOp = $value;})*;

relationalOperation returns [RightHandSideExpression value]
	: op2 = additiveOperation {$value = $op2.value; RightHandSideExpression lastOp = $value; OperationType type = null;}
	((LESS op2 = additiveOperation {type = OperationType.LESS;}
	|LESS_EQUAL op2 = additiveOperation {type = OperationType.LESS_EQUAL;}
	|GREATER op2 = additiveOperation {type = OperationType.GREATER;}
	|GREATER_EQUAL op2 = additiveOperation {type = OperationType.GREATER_EQUAL;})
	{$value = createBinaryOperation(lastOp, type, op2); lastOp = $value;})*;
	
additiveOperation returns [RightHandSideExpression value]
	:{OperationType type = null;} 
	(op2 = multiplicativeOperation {$value = $op2.value; RightHandSideExpression lastOp = $value;}
	((PLUS op2 = multiplicativeOperation {type = OperationType.ADDITION;}
	|MINUS op2 = multiplicativeOperation {type = OperationType.SUBTRACTION;})
	{$value = createBinaryOperation(lastOp, type, op2); lastOp = $value;})*);
	
multiplicativeOperation returns [RightHandSideExpression value]
	: op2 = unaryOperation {$value = $op2.value; RightHandSideExpression lastOp = $value; OperationType type = null;}
    ((MULT op2 = unaryOperation{type = OperationType.MULTIPLICATION;}
    |DIV op2 = unaryOperation{type = OperationType.DIVISION;}
    |MOD op2 = unaryOperation{type = OperationType.MODULO;})
    {$value = createBinaryOperation(lastOp, type, op2); lastOp = $value;})*;
    
unaryOperation returns [RightHandSideExpression value] 
	: (MINUS lq=literalOrQualifier {$value = createUnaryOperation(lq,OperationType.NEGATION);}
	| NOT lq=literalOrQualifier {$value = createUnaryOperation(lq,OperationType.NOT);}
	| lq=literalOrQualifier {$value = lq;});

literalOrQualifier returns [RightHandSideExpression value]
	: (q=literal {$value = $q.value;} 
	| rhs=parExpression {$value = $rhs.value;}
	| lhs=leftHandExp[this.currentScope] {$value = $lhs.value;})
	;


literal returns [Literal value]
	: (p=numberLiteral{$value = $p.value;}
	| s=stringLiteral{$value = $s.value;}
	| b=definedLiteral {$value = $b.value;}
	);
	// TODO: parse date literals

numberLiteral returns [Literal value]
	: (number)+ {$value = ISMFactory.eINSTANCE.createLiteral(); 
	$value.setType(TypeResolver.getNumberType());
	$value.setValue(Integer.parseInt($text));}
	(DOT (number)* {$value.setValue(Double.parseDouble($text));
	$value.setType(TypeResolver.getRealType());})?	
	;

stringLiteral returns [Literal value] 
	: e = STRING
    {$value = ISMFactory.eINSTANCE.createLiteral();
    $value.setType(TypeResolver.getStringType());
    $value.setValue($e.text);};
definedLiteral returns [Literal value]
	:{$value = ISMFactory.eINSTANCE.createLiteral();}
	( NULL {$value.setType(TypeResolver.getNullType());}
	| TRUE {$value.setValue(true);$value.setType(TypeResolver.getBooleanType());}
	| FALSE {$value.setValue(false);$value.setType(TypeResolver.getBooleanType());}
	)
	;

STRING          
@init{StringBuilder lBuf = new StringBuilder();}
    :   
           ('"' 
           ( escaped=ESC {lBuf.append(escaped.getText());} | 
             normal=~('"'|'\\'|'\n'|'\r')     {lBuf.appendCodePoint(normal);} )* 
           '"'
           |'\''
           ( escaped=ESC {lBuf.append(escaped.getText());} | 
             normal=~('\''|'\\'|'\n'|'\r')     {lBuf.appendCodePoint(normal);} )*
           '\''
           )
           {setText(lBuf.toString());}
    ;

fragment
ESC
    :   '\\' 
        (       'n'    {setText("\n");}
        |       'r'    {setText("\r");}
        |       't'    {setText("\t");}
        |       'b'    {setText("\b");}
        |       'f'    {setText("\f");}
        |       '"'    {setText("\"");}
        |       '\''   {setText("\'");} 
        |       '/'    {setText("/");}
        |       '\\'   {setText("\\");}
        |       ('u')+ HEX_DIGIT HEX_DIGIT HEX_DIGIT HEX_DIGIT
        )
    ;

string returns [String value]
	:( '"' (~'"')* '"'
	| '\'' .* '\'')
	{$value = $text;}
	;
/*
fragment StringChar 
	:  UnicodeChar | EscapeSequence
	;
fragment EscapeSequence
	: '\\' ('\"' | '\\' | '/' | 'b' | 'f' | 'n' | 'r' | 't' | 'u' HexDigit HexDigit HexDigit HexDigit)
	;
fragment UnicodeChar
	: ~('"'| '\\')
	;
*/
fragment HEX_DIGIT 
	: ('0'..'9'|'a'..'f'|'A'..'F') 
	;



/** some arithmetic operators. 
 * TODO: add further functions and operators for handling lists */
operator returns [OperationType value]	: PLUS {$value=OperationType.ADDITION;} 
										| MINUS {$value=OperationType.SUBTRACTION;}
										| MULT {$value=OperationType.MULTIPLICATION;}
										| DIV {$value=OperationType.DIVISION;};

/* type/class name */
typeName returns [Type value]
	:	ID (ARRAY|SET)? {$value = resolveType($text);};

/* name used in scopes, variables and properties */
name returns [String value]
	: 	ID {$value = $text;};

/*
fragment
letter	:	(SML_LETTER | CAP_LETTER) ;
fragment
character	:	letter | number | SIGN;
fragment
startChar	:	SML_LETTER | SIGN;*/

fragment
number 	:	ZERO_TO_NINE;

/*********************************************
	The Lexer section:
**********************************************/

/* Operators*/
ASSIGN	
	:	':=';
PLUS	
	:	'+';
MINUS	
	:	'-';
MULT	
	:	'*';
DIV		
	:	'/';
MOD		
	:	'%';
OR	
	:	'||';
OR2		
	:	'or';
AND
	:	'&&';
AND2		
	:	'and';
NOT	
	:	'!';
EQUALS	
	:	'=';
EQUALS2
	:	'==';
NOT_EQUALS 
	: '!=';
LESS	
	:	'<'; 
LESS_EQUAL	
	:	'<=';
GREATER	
	:	'>';
GREATER_EQUAL
	:	'>=';
DOT
	:	'.';
SEMICOLON
	:	';';
COLON
	:	':';
SCOPE_REF
	:	'::';
SIGN
	:	'_';
ARRAY
	:	'[]';
SET
	:	'{}';
SINGLE_QUOTE
	:	'\'';
UMLAUTS
	:	'ä'|'ö'|'ü'|'Ä'|'Ö'|'Ü'|'ß';

NULL : 'null';

TRUE : 'true';

FALSE : 'false';
	
/* Some letter and number ranges and newline definitions */
ID	:	('a'..'z' | 'A'..'Z'| SIGN)('a'..'z' | 'A'..'Z' | '0'..'9' | SIGN)*;
ZERO_TO_NINE	:	'0'..'9';

/* ignore whitespaces and new lines */
WS  :  (' '|'\r'|'\t'|'\u000C'|'\n') {$channel=HIDDEN;}
    ;

COMMENT /* ignore comments (like this one) */ 
    :   '/*' .* '*/' {$channel=HIDDEN;}
    ;
