/*
 * This file has been developed at the University of Munich, Chair for Programming & Software Engineering.
 * 
 * This file is licensed under the Eclipse Public License (EPL) 1.0
 * 
 */
package eu.mdd4soa.trans.uml2ism.test;

import static org.junit.Assert.assertEquals;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.List;
import java.util.Map;

import org.eclipse.uml2.uml.Activity;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Model;

import eu.mdd4soa.smm.exception.InvalidSetupException;
import eu.mdd4soa.smm.statik.Participant;
import eu.mdd4soa.trans.uml2smm.UMLEMFUtil;
import eu.mdd4soa.trans.uml2smm.transform.BehaviourConverter;
import eu.mdd4soa.trans.uml2smm.transform.StaticConverter;

public abstract class AbstractUML2SMMTest {

	private static Map<Class, List<Activity>> fParticipants;

	protected static void setUp(String fileName) throws InvalidSetupException {
		Model model= UMLEMFUtil.loadUMLModelFromFile(new File(new File("."), fileName));
		fParticipants= UMLEMFUtil.getParticipantsAndActivities(model);
	}

	protected static Class getParticipant(String name) {
		for (Class p : fParticipants.keySet()) {
			if (p.getName().equals(name))
				return p;
		}
		return null;
	}

	protected static Activity getActivity(Class p, String activityName) {

		List<Activity> list= fParticipants.get(p);
		for (Activity activity : list) {
			if (activity.getName().equals(activityName))
				return activity;
		}

		return null;
	}

	protected static Activity getActivity(String name) throws Exception {

		for (List<Activity> la : fParticipants.values()) {
			for (Activity activity : la) {
				if (activity.getName().equals(name))
					return activity;
			}
		}

		throw new Exception("Activity with name " + name + " not found.");
	}

	protected void checkStatic(String partName) throws Exception {

		Class p= getParticipant(partName);
		if (p == null)
			throw new Exception("Could not find participant: " + partName);


		Participant ismConverted= StaticConverter.convert(p);

		// Check it.
		checkISM(ismConverted, "Static_" + partName);
	}

	protected void checkOrch(String partName, String activityName) throws Exception {

		Class p= getParticipant(partName);
		if (p == null)
			throw new Exception("Could not find participant: " + partName);
		Activity activity= getActivity(p, activityName);
		if (activity == null)
			throw new Exception("Could not find activity: " + activityName + " in participant " + partName);


		Participant ismConverted= StaticConverter.convert(p);
		BehaviourConverter.convert(ismConverted, activity);

		// Check it.
		checkISM(ismConverted, partName + "_" + activityName);
	}


	protected void checkISM(Participant c, String fileName) throws Exception {
		File justNewlyGenerated= new File(new File("."), "resources/temp/" + fileName + ".xmi");
		justNewlyGenerated.deleteOnExit();

		UMLEMFUtil.storeSMM(justNewlyGenerated, c);
		File theCorrectFile= new File(new File("."), "resources/expected/" + fileName + ".xmi");

		String correctString= loadIntoString(theCorrectFile);
		String generatedString= loadIntoString(justNewlyGenerated);

		justNewlyGenerated.delete();
		assertEquals("Content must match", correctString, generatedString);
	}

	private String loadIntoString(File fileToTest) throws Exception {

		if (!fileToTest.exists())
			throw new Exception("Could not find the file: " + fileToTest);

		BufferedReader fileToTestReader= new BufferedReader(new FileReader(fileToTest));
		StringBuilder completeFileToTest= new StringBuilder();
		String fileToTestLine= null;
		while ( (fileToTestLine= fileToTestReader.readLine()) != null) {
			completeFileToTest.append(fileToTestLine + "\n");
		}
		fileToTestReader.close();

		return completeFileToTest.toString();
	}


}
