/*
 * This file has been developed at the University of Munich, Chair for Programming & Software Engineering.
 * 
 * This file is licensed under the Eclipse Public License (EPL) 1.0
 * 
 */
package eu.uml4soa.utbm.u2m.test;

import static org.junit.Assert.assertEquals;

import org.junit.Test;

import eu.mdd4soa.smm.behaviour.ServiceActivity;
import eu.uml4soa.utbm.u2m.SemanticsException;
import eu.uml4soa.utbm.u2m.model.CAutomaton;
import eu.uml4soa.utbm.u2m.semantics.OrchestrationSemanticsFunction;

/**
 * This class tests the basic interaction features like send, receive,
 * send&receive in the conversion.
 * 
 * @author Philip Mayer, mayer@pst.ifi.lmu.de
 * 
 */
public class TestInteraction extends AbstractSemanticTests {

	@Test
	public void testInteractionSnd() throws SemanticsException {

		ServiceActivity sa= createActivity("Test Snd");
		sa.getChildren().add(createSend("performSth"));

		CAutomaton c= OrchestrationSemanticsFunction.transform(sa);
		assertEquals("Must have two states", 2, c.getStates().size());
		assertEquals("Must have one transition", 1, c.getTransitions().size());
		assertEquals("End state must have 1 end state", 1, c.getNonErrorEndStates().size());

		assertEquals("End state must be end snd performSth(1)", "end snd performSth(1)", getFirstAndOnly(c.getNonErrorEndStates()).getName());
	}

	@Test
	public void testInteractionTwoSnds() throws SemanticsException {

		ServiceActivity sa= createActivity("Test Two Snds");
		sa.getChildren().add(createSend("performSth"));
		sa.getChildren().add(createSend("doItAgainSth"));

		CAutomaton c= OrchestrationSemanticsFunction.transform(sa);
		assertEquals("Must have three states", 3, c.getStates().size());
		assertEquals("Must have two transitions", 2, c.getTransitions().size());
		assertEquals("End state must have 1 end state", 1, c.getNonErrorEndStates().size());

		assertEquals("End state must be end snd doItAgainSth(2)", "end snd doItAgainSth(2)", getFirstAndOnly(c.getNonErrorEndStates()).getName());
	}

	@Test
	public void testInteractionReceive() throws SemanticsException {

		ServiceActivity sa= createActivity("Test One Rcv");
		sa.getChildren().add(createReceive("inputComesHere"));

		CAutomaton c= OrchestrationSemanticsFunction.transform(sa);
		assertEquals("Must have two states", 2, c.getStates().size());
		assertEquals("Must have one transition", 1, c.getTransitions().size());
		assertEquals("End state must have 1 end state", 1, c.getNonErrorEndStates().size());

		assertEquals("End state does not match!", "end rcv inputComesHere(1)", getFirstAndOnly(c.getNonErrorEndStates()).getName());
	}

	@Test
	public void testInteractionReceiveAndReply() throws SemanticsException {

		ServiceActivity sa= createActivity("Test Rcv/Rply");
		sa.getChildren().add(createReceive("request"));
		sa.getChildren().add(createReply("request"));

		CAutomaton c= OrchestrationSemanticsFunction.transform(sa);
		assertEquals("Must have three states", 3, c.getStates().size());
		assertEquals("Must have two transitions", 2, c.getTransitions().size());
		assertEquals("End state must have 1 end state", 1, c.getNonErrorEndStates().size());

		assertEquals("Start state does not match!", "start rcv request(1)", c.getStart().getName());
		assertEquals("End state does not match!", "end reply request(2)", getFirstAndOnly(c.getNonErrorEndStates()).getName());
	}

	@Test
	public void testInteractionSendAndReceive() throws SemanticsException {

		ServiceActivity sa= createActivity("Test Rcv/Snd");
		sa.getChildren().add(createSendAndReceive("request"));

		CAutomaton c= OrchestrationSemanticsFunction.transform(sa);
		assertEquals("Must have three states", 3, c.getStates().size());
		assertEquals("Must have two transition", 2, c.getTransitions().size());
		assertEquals("End state must have 1 end state", 1, c.getNonErrorEndStates().size());

		assertEquals("Start state does not match!", "start snd/rcv request(1)", c.getStart().getName());
		assertEquals("End state does not match!", "end snd/rcv request(1)", getFirstAndOnly(c.getNonErrorEndStates()).getName());
	}

	@Test
	public void testInteractionData() throws SemanticsException {

		ServiceActivity sa= createActivity("Test Data");
		sa.getChildren().add(createData("assign"));

		CAutomaton c= OrchestrationSemanticsFunction.transform(sa);
		assertEquals("Must have three states", 2, c.getStates().size());
		assertEquals("Must have two transition", 1, c.getTransitions().size());
		assertEquals("End state must have 1 end state", 1, c.getNonErrorEndStates().size());

		checkStateMatch("Start state does not match!", "start data assign", c.getStart());
		checkStateMatch("End state does not match!", "end data assign", getFirstAndOnly(c.getNonErrorEndStates()));
	}

}
